//; Installation package was created by InstallationPackageBuilder 1.4.0.9

[Code]

const 
  csServisInstallError = 'Installation of "%s" fails with error code: "%d", message: "%s". Please try installation again and if the issue persists, contact our technical support for help.';
  csServisUnInstallError = 'Uninstallation of "%s" fails with error code: "%d", message: "%s". Please try installation again and if the issue persists, contact our technical support for help.';
  cServiceStateChangeWaitingCycles = 3000;
  cServiceStateChangeDelay = 200; // [ms]
  cUninstallRegKey = 'Software\Microsoft\Windows\CurrentVersion\Uninstall\';

type
  TUninstallPreviousVersionResult = (
    uovrOK,    // successfully executed the UnInstallString
    uovrError, // error executing the UnInstallString
    uovrSkip   // uninstall was skipped
    );

var
  sTodaysName: String;
  sInstalledVersion: String;
  DefaultSurfaceLeft: Integer;
  DefaultSurfaceTop: Integer;
  DefaultSurfaceWidth: Integer;
  DefaultSurfaceHeight: Integer;
  ImagePageId: Integer;
  SurfaceIsChanged: Boolean;
  RestartIsNeeded: Boolean;
  PsiPackageCanRun: Boolean;
  bAddWizardImagePackage: Boolean;
  gSetupLogDir: String;
  ForceClose: Boolean;

procedure LogRegistry(ARegistry: String);
begin
  StringChangeEx(ARegistry, '''''', '"', True);
  Log(Format('Write to registry: %s', [ ExpandConstant(ARegistry) ]));
end;

{ Find parametr index in commmand line, otherwise returns -1 }
function FindCmdLineSwitchIndex(const ASwitch: String): Integer;
var
  I: Integer;
  hlpParam: string;
begin
  for I := 1 to ParamCount do
  begin
    hlpParam := ParamStr(I);
    if (hlpParam[1] = '/') then
      if (CompareText(hlpParam, ASwitch) = 0) then
      begin
        Result := I;
        Exit;
      end;
  end;
  Result := -1;
end;

{ Returns value according to the switch (this is comap way, returns just next one) }
function CmdLineSwitchValue(const ASwitch: String): String;
var
  iIdx: Integer;
begin
  iIdx := FindCmdLineSwitchIndex(ASwitch);
   // find index of switch
  { the next one conatins the value we expect }
  if (iIdx >= 0) and (ParamCount > iIdx) then
  begin
    Result := ParamStr(iIdx + 1);
  end
  else
  begin
    Result := '';
     // no value at all
  end;
end;

{ Returns path to desktop folder base on selected user group - current user or public }
function GetSelectedDesktopFolder(Param: String): String;
begin
  if IsTaskSelected('userinstall') then
    Result := ExpandConstant('{userdesktop}')
  else
    Result := ExpandConstant('{commondesktop}');
end;

{ returns registry key value according to the publisher name. Could be pass by main package as parametr }
function GetRegKeyPublisher(): String;
begin
  Result := CmdLineSwitchValue('/regpub');
  if (Result = '') then
    Result := 'ComAp';
end;

{ returns registry key value according to the pc suite name. Could be pass by main package as parametr }
function GetRegKeyPcSuiteName(): String;
begin
  Result := CmdLineSwitchValue('/regpcsuite');
  if (Result = '') then
    Result := 'ComAp PC Suite';
end;

function GetPcSuiteRegKey(ASuffixPath: String): String;
begin
  if (GetRegKeyPcSuiteName() <> GetRegKeyPublisher()) then
    Result := 'Software\' + GetRegKeyPublisher() + '\' + GetRegKeyPcSuiteName()
  else
    Result := 'Software\' + GetRegKeyPublisher();
  if (ASuffixPath <> '') then
    Result := Result + '\' + ASuffixPath;
end;
 
function GetPcSuiteRegKey64(ASuffixPath: String): String;
begin
  if (GetRegKeyPcSuiteName() <> GetRegKeyPublisher()) then
    Result := 'Software\Wow6432Node\' + GetRegKeyPublisher() + '\' + GetRegKeyPcSuiteName()
  else
    Result := 'Software\Wow6432Node\' + GetRegKeyPublisher();
  if (ASuffixPath <> '') then
    Result := Result + '\' + ASuffixPath;
end;

function GetPublisherRegKey(ASuffixPath: String): String;
begin
  Result := 'Software\' + GetRegKeyPublisher();
  if (ASuffixPath <> '') then
    Result := Result + '\' + ASuffixPath;
end;

function GetPublisherRegKey64(ASuffixPath: String): String;
begin
  Result := 'Software\Wow6432Node\' + GetRegKeyPublisher();
  if (ASuffixPath <> '') then
    Result := Result + '\' + ASuffixPath;
end;

function GetInstalledAppVersion(AppName: String): String;
 // returns current instaled version of program application
 // not installed returns ''
var
  sVersion: String;
  hlpAppName: String;
begin
  Result := '';
  if AppName = '' then
    hlpAppName := 'ComAp PC Suite'
  else
    hlpAppName := AppName;
  //if RegQueryStringValue(HKLM, 'Software\ComAp\ComAp PC Suite\{#APP_NAME}', 'Version', sVersion) then
  if RegQueryStringValue(HKLM, GetPcSuiteRegKey(hlpAppName), 'Version', sVersion) then
    Result := sVersion
  //else if RegQueryStringValue(HKLM, 'Software\Wow6432Node\ComAp\ComAp PC Suite\{#APP_NAME}', 'Version', sVersion) then
  else if RegQueryStringValue(HKLM, GetPcSuiteRegKey64(hlpAppName), 'Version', sVersion) then
    Result := sVersion;
end;

function GetInstalledVersion(): String;
 // returns current installed version of program
 // not installed returns ''
begin
    Result := GetInstalledAppVersion('');
end;

{ this procedure convert version to int but in ComAp way - experimental build number 900-999 is decreased by 1000}
procedure VersionToInt(AVersion: String; out A1: Integer; out A2: Integer; out A3: Integer; out A4: Integer);
var
  iPart, iPos, iNumb: Integer;
  sNumb: String;
begin
  for iPart := 0 to 3 do
  begin
    iPos := Pos('.', AVersion);
    if iPos = 0 then
    begin
      if (AVersion <> '') then
      begin
        sNumb := AVersion;
        AVersion := '';
      end
      else
      begin
        sNumb := '0';
      end;
    end
    else
    begin
      sNumb := Copy(AVersion, 1, iPos - 1);
      Delete(AVersion, 1, iPos);
    end;
    try
      iNumb := StrToInt(sNumb);
    except
      iNumb := -1;
    end;
    case iPart of
      0: A1 := iNumb;
      1: A2 := iNumb;
      2: A3 := iNumb;
      3: A4 := iNumb;
    end;
  end;
  if A4 >= 900 then
    A4 := A4 - 1000;
   // COMAPPCSUITE-147 Experimental builds (build number 900+) are counted lower then regular builds
end;

{ returns directory contains application (if exist) or return fake directory 
  for safety because it will be used for directory deletion }
function GetProgramDirForDeletion(APar: string): string;
begin
  if (Pos(LowerCase('ComAp PC Suite'), LowerCase(ExpandConstant('{app}'))) > 0) then
    Result := ExpandConstant('{app}')
  else
    Result := 'c:\fake\dir\which\doesnt\exist\or\can\be\deleted\i\hope';
end;

{ compares AVersion1 and AVersion2
  returns < than 0 if AVersion1 < AVersion2;
  returns 0 if AVersion1 = AVersion2
  returns > than 0 if AVersion1 > AVersion2 }
function CompareFilesVersions(AVersion1, AVersion2: String): Integer;
var
  hlpVer1: array[0..3] of Integer;
  hlpVer2: array[0..3] of Integer;
  i: Integer;
begin
  VersionToInt(AVersion1, hlpVer1[0], hlpVer1[1], hlpVer1[2], hlpVer1[3]);
   // get version 1 -> 0 is most significant, 3 less significant
  VersionToInt(AVersion2, hlpVer2[0], hlpVer2[1], hlpVer2[2], hlpVer2[3]);
   // get version 2
  { iterate thru all versins part }
  for i:=0 to 3 do
  begin
    Result := hlpVer1[i] - hlpVer2[i];
     // subtract versions parts with same significancy  
     // if V1 < V2 returns number lower than 0
     // if V1 > V2 returns number greather than 0
     // if V1 = V2 returns 0
    { if the result is not 0, we can skip other parts of versions }
    if Result <> 0 then
      Exit;
  end;
end;

procedure LogExecResult(ResultCode: Integer; IsSuccessful: Boolean);
begin
  if (IsSuccessful) then
    Log('Exec result: result code ' + IntToStr(ResultCode))
  else
    Log('Exec result: Failed with message: ' + SysErrorMessage(ResultCode) + ' (code=' + IntToStr(ResultCode) + ').');
end;

{ Updates the setup log file after a file is installed. Adds SHA1 for the file. 
  JIRA: COMAPPCSUITE-170 }
procedure LogFileHash();
begin
  Log('SHA1: ' + GetSHA1OfFile(ExpandConstant(CurrentFileName())));
end;

{ Returns True when installation is running in Silent or VerySilent mode }
function IsSilentOrVerySilentMode: Boolean;
begin
  Result := ((FindCmdLineSwitchIndex('/VERYSILENT') <> -1) or (FindCmdLineSwitchIndex('/SILENT') <> -1));
end;

{ Disable installation of individual subpackage. It is expected that such an installation is invalid within our installation package system. }
function CheckSeparateInstallation(): Boolean;
begin
  if (IsSilentOrVerySilentMode) then
  begin
    Result := True;
  end
  else
  begin
    SuppressibleMsgBox(CustomMessage('OnlyAsInstallSuite'), mbInformation, MB_OK, IDOK);
    Result := False;
  end;
end;

{ Disable installation on old Windows. }
function CheckWindowsVersion(MinimalSupportedWindowsVersion: Cardinal; Message: String): Boolean;
begin
  if (GetWindowsVersion() >= MinimalSupportedWindowsVersion) then
  begin
    Result := True;
  end
  else
  begin
    SuppressibleMsgBox(Message, mbInformation, MB_OK, IDOK);
    Result := False;
  end;
end;

function CheckActualVersion(): Boolean;
var
  sActualVersion: String;
begin
  Result := True;
  sInstalledVersion := GetInstalledVersion;
   // to fing out the installed version 
  { if we have info about installed vestion }
  if sInstalledVersion <> '' then
  begin
    Log('Currently installed version of ComAp PC Suite: ' + sInstalledVersion)
    sActualVersion := '4.1.5.2';
    if (ExpandConstant('{param:AlwaysOverride|false}') = 'true') then Exit; // Result = true
    { installed version is older than currently installed }
    if CompareFilesVersions(sActualVersion, sInstalledVersion) < 0 then
    begin
      if (ExpandConstant('{param:KeepNewerVersion|false}') = 'true') or (SuppressibleMsgBox(FmtMessage(ExpandConstant('{cm:WouldYouLikeToReplace}'), ['ComAp PC Suite', sInstalledVersion, '4.1.5.2']), mbConfirmation, MB_YESNO, IDYES) = IDNO) then
      begin
        Result := False;
      end;
    end;
  end;
end;

// SharedPascalFunctions.iss
{returns true if setup runs with switch in parameter}
function IsRunningWithSwitch(ASwitch: String): Boolean;
var
  paramIndex: Integer;
begin
  Result := False;
  for paramIndex := 0 to ParamCount() do
    if CompareText(ParamStr(paramIndex), ASwitch) = 0 then
    begin
      Result := True;
      Exit;
    end;
end;


procedure BackUpDir(ADir: String);
var
  sNewName: String;
begin
  sNewName := ADir + ' - BackUp ' + GetDateTimeString('yyyy/mm/dd hh.nn.ss', '-', #0);
  if not RenameFile(ADir, sNewName) then
  begin
    // nothing to report
  end;
end;

procedure BackUpUserDirs();
begin
  BackUpDir(ExpandConstant('{userappdata}') + '\ComAp PC Suite');
  BackUpDir(ExpandConstant('{userdocs}') + '\ComAp PC Suite');
end;

procedure SortStartMenuFolderByName(AName: String);
 // order start menu folder by name
begin
  RegDeleteValue(HKCU, 'Software\Microsoft\Windows\CurrentVersion\Explorer\MenuOrder\Start Menu\Programs\' + AName, 'Order');
end;

{ Add wizard image if wizard width is between AMinWidth and AMaxWidth }
procedure AddWizardImagePage(AFileName: string; AMinWidth: Integer; AMaxWidth: Integer);
var 
  page: TWizardPage;
  image: TBitmapImage;
begin  
  if ((WizardForm.ClientWidth >= AMinWidth) and (WizardForm.ClientWidth <= AMaxWidth)) then
  begin  
    ExtractTemporaryFile(AFileName);
    page := CreateCustomPage(wpWelcome, '', '');
    DefaultSurfaceLeft := page.Surface.Notebook.Left;
    DefaultSurfaceTop := page.Surface.Notebook.Top;
    DefaultSurfaceWidth := page.Surface.Notebook.Width;
    DefaultSurfaceHeight := page.Surface.Notebook.Height;
    ImagePageId := page.ID;

    image := TBitmapImage.Create(page);
    image.Parent := page.Surface;
    image.SetBounds(0, 0, WizardForm.ClientWidth, WizardForm.Bevel.Top);
    image.Bitmap.LoadFromFile(ExpandConstant('{tmp}') + '\' + AFileName);
    image.Stretch := True;
  end;
end;

procedure SetBoundsForPage(CurPageID: Integer);
var
  page: TWizardPage;
begin
  {change size of surface for image page}
  page := PageFromID(CurPageID);
  if (CurPageID = ImagePageID) then
  begin
    page.Surface.Notebook.SetBounds(0, 0, WizardForm.ClientWidth, WizardForm.Bevel.Top);  
    page.Surface.Notebook.BringToFront();
  end;
  if (CurPageID = wpLicense) then
  begin
    page.Surface.Notebook.SetBounds(DefaultSurfaceLeft, DefaultSurfaceTop, DefaultSurfaceWidth, DefaultSurfaceHeight);  
    page.Surface.Notebook.SendToBack();
  end;
end;

// Gets path for unistall app
function GetUninstallString(): String;
var
  sAppId: String;
  sUnInstallString: String;
  sUnInstPath: String;
begin
  sAppId := '{{347EEEAE-0F1E-4823-BE31-E3F0C6B8960C}';
  sAppId := Copy(sAppId, 2, Length(sAppId) - 1);
  sUnInstPath :=  cUninstallRegKey + sAppId + '_is1';
  sUnInstallString := '';

  if not RegQueryStringValue(HKLM, sUnInstPath, 'UninstallString', sUnInstallString) then
    RegQueryStringValue(HKCU, sUnInstPath, 'UninstallString', sUnInstallString);
  Result := sUnInstallString;
end;

// Uninstall previous application version
function UnInstallPreviousVersion(): TUninstallPreviousVersionResult;
var
  sUnInstallString: String;
  iResultCode: Integer;
begin
  sUnInstallString := GetUninstallString(); // get the uninstall string of the previous app
  if sUnInstallString <> '' then
  begin
	sUnInstallString := RemoveQuotes(sUnInstallString);
    if Exec(sUnInstallString, '/SILENT /NORESTART /SUPPRESSMSGBOXES','', SW_HIDE, ewWaitUntilTerminated, iResultCode) then
      Result := uovrOK
    else
      Result := uovrError;
  end
  else
    Result := uovrSkip;
end;

// Checks running process. Returns True when process is running otherwise False.
// AProcessName - EXE file name without path
function ProcessRunning(AProcessName : String): Boolean;
var
  tmpLocator, tmpWMIService, tmpObjectSet: Variant;
begin
  Result := False;
  try
    // The problem is described here: https://docs.microsoft.com/cs-cz/windows/desktop/CIMWin32Prov/terminate-method-in-class-win32-process
	tmpLocator := CreateOleObject('WBEMScripting.SWBEMLocator');
    tmpWMIService := tmpLocator.ConnectServer('', 'root\CIMV2', '', '');
    tmpObjectSet := tmpWMIService.ExecQuery(Format('SELECT Name FROM Win32_Process Where Name="%s"',[AProcessName]));
    Result := (tmpObjectSet.Count > 0);
  finally
    tmpObjectSet := Unassigned;
    tmpWMIService := Unassigned;
    tmpLocator := Unassigned;	
  end;
end;

// Checks when process is running. If process is running, then user will be informed or for silent mode information will be added only to log.
procedure CheckRunningProcess(sFileName, sApplicationName: String);
var 
  iMsgResult: Integer;
begin
  SuppressibleMsgBox(CustomMessage('CheckRunningComApApps'), mbInformation, MB_OK, IDOK);

  while (ProcessRunning(sFileName)) do
    begin
      Log(Format('WARNING: Application %s is running.', [sApplicationName]));
	  if (IsSilentOrVerySilentMode) then
        Exit
      else
	  begin
        iMsgResult := SuppressibleMsgBox(FmtMessage(CustomMessage('ApplicationRunning'), [sApplicationName]), mbError, MB_OKCANCEL, IDOK);
        if iMsgResult = IDCANCEL then
		  Abort;
	  end;
    end;
end;

// Check whether Persistent Storage Service is installed and PSI extension is associated by Persistent Storage Importer
function PsiCanBeRun(): Boolean;
var
  sPersistentStorageServiceExe, sPsiApp, sPsiAppShell, sPsiAppShellOrig, sArgument: String;
  bPsiAssociationError: Boolean;
  iPathExePosition, iPathExePositionOrig: Integer;
begin
  sPersistentStorageServiceExe := 'ComAp.PersistentStorage.PersistentStorageService.exe';
  bPsiAssociationError := True;

  if (RegKeyExists(HKCR, '.psi')) then
  begin
	RegQueryStringValue(HKCR, '.psi', '', sPsiApp);	
	if (sPsiApp <> '') and (RegKeyExists(HKCR, sPsiApp +'\shell\open\command')) then
	begin
	  RegQueryStringValue(HKCR, sPsiApp +'\shell\open\command', '', sPsiAppShellOrig);
	  sPsiAppShell := RemoveQuotes(sPsiAppShellOrig);
	  iPathExePosition := Pos('.exe', sPsiAppShell);
	  if (iPathExePosition > 0) then
	  begin
		iPathExePosition := iPathExePosition + Length('.exe') - 1;
		iPathExePositionOrig := Pos('.exe', sPsiAppShellOrig) + Length('.exe"');
		sArgument := Copy(sPsiAppShellOrig, iPathExePositionOrig, Length(sPsiAppShellOrig) - iPathExePositionOrig + 1);
		sPsiAppShell := Copy(sPsiAppShell, 0, iPathExePosition);
	  end;
	  if (sPsiAppShell <> '') and 
		 (FileExists(sPsiAppShell)) and 
		 (ExtractFileName(sPsiAppShell) = 'ComAp.PersistentStorage.PersistentStorageImport.exe') and
		 (sArgument = ' -p "%1"') then
		 bPsiAssociationError := False;
	  end;
	end;

	if PsiPackageCanRun and bPsiAssociationError then
	begin
	  Log('ERROR: PSI files do not associate with ComAp PersistentStorage Importer!');
	  SuppressibleMsgBox(CustomMessage('PsiFilesNotAssociate'), mbCriticalError, MB_OK, IDOK);
	  PsiPackageCanRun := False;
	end;

	if PsiPackageCanRun and (not ProcessRunning(sPersistentStorageServiceExe)) then
	begin
	  Log('ERROR: ComAp PersistentStorage service is not installed or is not running!');
	  SuppressibleMsgBox(CustomMessage('PersistentstorageServiceIsNotInstalled'), mbCriticalError, MB_OK, IDOK);
	  PsiPackageCanRun := False;
	end;
	
	Result := PsiPackageCanRun;
end;

/// <summary> Check that /INTERNALUSE switch was passed to package.</summary>
function IsInternalUseSwitchFound(): Boolean;
begin
  if FindCmdLineSwitchIndex('/INTERNALUSE') <> -1 then
    Result := True
  else
    Result := False;
end;

{ Sets new pages with specific image }
procedure AddWizardImagePages;
begin
  

  bAddWizardImagePackage := True;
end;

{ Shows wait dialog for running AddWizardImagePages procedure }
procedure ShowWaitDialog;
var
  dlgWait : TSetupForm;
  lblWait : TLabel;
begin
  dlgWait :=  CreateCustomForm;
  dlgWait.FormStyle := bsToolWindow;
  dlgWait.Position := poScreenCenter;
  dlgWait.Width := 200;
  dlgWait.Height := 100;
  dlgWait.Caption := FmtMessage(CustomMessage('SetupWindowTitle'), ['ComAp PC Suite']);

  lblWait := TLabel.Create(dlgWait);
  lblWait.Parent := dlgWait;
  lblWait.Caption := CustomMessage('PleaseWait');
  lblWait.Visible := True;
  lblWait.Width := (200 * 100) / 100;
  lblWait.Height := (60 * 100) / 100;
  lblWait.Alignment := taCenter;
  lblWait.Layout := tlCenter;
  dlgWait.Show;
  dlgWait.Refresh; // Process the paint message

  { Actions }
  AddWizardImagePages;
  
  dlgWait.Free;
end;

function GetSetupLogDir(): String;
begin
  if (gSetupLogDir = '') then
  begin
    gSetupLogDir := ExpandConstant('{commonappdata}') + '\ComAp PC Suite\Setup Logs\' + 
      GetDateTimeString('yyyymmddhhnnss', #0, #0) + '_InteliMonitor-Product-Suite-4.1.5.2\';
    ForceDirectories(gSetupLogDir);
  end;
  Result := gSetupLogDir; 
end;

function GetSetupLogDirProxy(Param: String): String;
begin
  Result := GetSetupLogDir; 
end;

procedure BackupLogFile(SourceFilePath: String; DestFileName: String);
begin
  if (FileExists(SourceFilePath) = False) then
    Exit;

  if DirExists(GetSetupLogDir()) then
  begin
    FileCopy(SourceFilePath, GetSetupLogDir() + DestFileName, False);
  end;
end;

procedure BackupMainLogFile();
begin
  BackupLogFile(ExpandConstant('{log}'), 'InteliMonitor-Product-Suite-4.1.5.2.log');
end;

{ It force close installation window without confirmation window. }
procedure Exterminate;
begin
  ForceClose:= True;
  WizardForm.Close;  
end;

{ Override default behavior of CancelButtonClick event. }
procedure CancelButtonClick(CurPageID: Integer; var Cancel, Confirm: Boolean);
begin
  Confirm:= not ForceClose;
end;

{ Get free space size of disk selected by path. }
function GetFreeSpaceSizeOfDisk(APath: String; AUnitsAreMegaBytes: Boolean): Cardinal;
var
  cFreeSpace, cTotalSpace: Cardinal;
begin
   if GetSpaceOnDisk(APath, AUnitsAreMegaBytes, cFreeSpace, cTotalSpace) then
   begin
      Result := cFreeSpace;
   end
   else
   begin
      Result := -1;
      Log(Format('Warning: Failed to check free space in driver with path: %s', [ APath ]));
   end;
end;

{ Check that size of package is lower than free space in disk. }
function PackageSizeCheck(ARequiredSpace: Integer): Boolean;
var
   idialogResponse: Integer;
   cFreeSpace : Cardinal;
   bUserChooseRetry : Boolean;
begin
   repeat
   bUserChooseRetry := False; 
   cFreeSpace := GetFreeSpaceSizeOfDisk(ExpandConstant('{app}'), True);
   if cFreeSpace < ARequiredSpace then
   begin
      idialogResponse := MsgBox(Format('Disc has not enough space for installation.'#13#10'Actual free space %d MB '#13#10'Require space %d MB', [ cFreeSpace, ARequiredSpace ]), mbConfirmation, MB_ABORTRETRYIGNORE);
      if idialogResponse = IDRETRY then
      begin
         bUserChooseRetry := True;
      end;

      if idialogResponse = IDABORT then
      begin
         Exterminate;
         Result := False;
      end;

      if idialogResponse = IDIGNORE then
      begin
         Result := True;
      end;
   end
   else
   begin
      Result := True;
   end;
   until not bUserChooseRetry;
end;

const
  csSetupLogFileName = 'setup.log';

  csFileNameLinePrefix = 'Dest filename: ';
  csConsistCheckResultFileName = 'installation-errors.log';
  csExludedFiles = 
      'log4net.config' + #0
      'ComAp.Communication.HostService.exe.config' + #0
      'ComAp.LicenceService.WinService.exe.config' + #0
      'ApmLicenceService.WinService.exe.config';

type

  TFileInfo = record
    Name : string;
    Path: string;
    Hash: string;
  end;

  TArrayOfFileInfo = array of TFileInfo;

// begin of consistencycheck
function InstalledFilesLog() : String;
begin
  Result := ExpandConstant('{tmp}\global-setup.log');
end;
  
procedure ProcessSetupLog(FilePath: String);
var
  hlpFileContent: TArrayOfString;
  hlpFilename, hlpLine: String;
  hlpGlobalLogFile: String;
  hlpPath: String;
  hlpPos, I: Integer;
begin
  hlpGlobalLogFile := InstalledFilesLog();
  if LoadStringsFromFile(FilePath, hlpFileContent) then
  begin
    for I := 0 to GetArrayLength(hlpFileContent) - 1 do
    begin
      hlpLine := hlpFileContent[I];
      hlpPos := Pos(csFileNameLinePrefix, hlpLine);
      if hlpPos <> 0 then
      begin
        hlpFilename := Copy(hlpLine, hlpPos + Length(csFileNameLinePrefix), Length(hlpLine));
        hlpPath := ExtractFilePath(hlpFilename);
        SaveStringToFile(hlpGlobalLogFile, hlpFilename + #13#10, True);
      end
    end;   
  end
  else
  begin
    Log('ConsistencyCheck: Cannot load file ' + FilePath);
  end;
end;

procedure FindInfoFiles(FileNames: TStringList);
var
  FindRec: TFindRec;
begin
  if FindFirst(ExpandConstant('{tmp}\*.info'), FindRec) then 
  begin
    try
      repeat
        FileNames.Add(FindRec.Name);
      until not FindNext(FindRec);
    finally
      FindClose(FindRec);
    end;
  end;
end;

procedure Explode(var Dest: TArrayOfString; Text: String; Separator: String);
var
  i, p: Integer;
begin
  i := 0;
  repeat
    SetArrayLength(Dest, i+1);
    p := Pos(Separator,Text);
    if p > 0 then 
    begin
      Dest[i] := Copy(Text, 1, p-1);
      Text := Copy(Text, p + Length(Separator), Length(Text));
      i := i + 1;
    end else begin
      Dest[i] := Text;
      Text := '';
    end;
  until Length(Text)=0;
end;

procedure ParseOriginalFilesInfo(FileName: String; var FilesInfo: TArrayOfFileInfo);
var
  hlpFileContent: TArrayOfString;
  hlpLineContent: TArrayOfString;
  I, hlpLength: Integer;
begin
  if LoadStringsFromFile(FileName, hlpFileContent) then
  begin
    // from index 1 because the first line contains column names
    for I := 1 to GetArrayLength(hlpFileContent) - 1 do
    begin
      hlpLength := GetArrayLength(FilesInfo);
      SetArrayLength(FilesInfo, hlpLength + 1);

      Explode(hlpLineContent, hlpFileContent[I], #9);
      FilesInfo[hlpLength].Name := ExtractFileName(hlpLineContent[1]);
      FilesInfo[hlpLength].Hash := hlpLineContent[4];
    end;  
  end
  else
  begin
    Log('ConsistencyCheck: Cannot load file ' + FileName);
  end;
end;

procedure ParseInstallFilesInfo(FileName: String; var FilesInfo: TArrayOfFileInfo);
var
  hlpFileContent: TArrayOfString;
  hlpFilename: String;
  I, hlpLength: Integer;
begin
  if LoadStringsFromFile(FileName, hlpFileContent) then
  begin
    // from index 1 because the first line contains column names
    for I := 1 to GetArrayLength(hlpFileContent) - 1 do
    begin
      hlpFileName := hlpFileContent[I]
      if FileExists(hlpFileName) then
      begin
        hlpLength := GetArrayLength(FilesInfo);
        SetArrayLength(FilesInfo, hlpLength + 1);
        FilesInfo[hlpLength].Name := ExtractFileName(hlpFileName);
        FilesInfo[hlpLength].Path := hlpFileName;
        FilesInfo[hlpLength].Hash := GetSHA1OfFile(hlpFileName);
      end;
    end;  
  end
  else
  begin
    Log('ConsistencyCheck: Cannot load file ' + FileName);
  end;
end;

function IsExcludedFromConsistencyCheck(FileName: String; ExcludedFiles: TArrayOfString) : Boolean;
var
  I: Integer;
begin
  Result := False;
  for I := 0 to GetArrayLength(ExcludedFiles) - 1 do
  begin
    if CompareText(FileName, ExcludedFiles[I]) = 0 then
    begin
      Result := True;
      Exit;
    end;
  end; 
end;

function FindFilesInfo(FileName: String; Files: TArrayOfFileInfo) : TArrayOfFileInfo;
var
  I, hlpLength: Integer;
begin
  SetArrayLength(Result, 0);
  for I := 0 to GetArrayLength(Files) - 1 do
  begin
    if CompareText(FileName, Files[I].Name) = 0 then
    begin
      hlpLength := GetArrayLength(Result);
      SetArrayLength(Result, hlpLength + 1);
      Result[hlpLength]:= Files[I];
    end;
  end;  
end;

procedure CompareFilesInfo(OriginalFiles: TArrayOfFileInfo; InstallFiles: TArrayOfFileInfo; CompareInfo: TStringList);
var
  hlpInstallFile: TFileInfo;
  hlpFiles: TArrayOfFileInfo;
  I, J: Integer;
  hlpFound: Boolean;
  hlpExcludedFiles: TArrayOfString;
begin
  Explode(hlpExcludedFiles, csExludedFiles, #0);
  for I := 0 to GetArrayLength(InstallFiles) - 1 do
  begin
    hlpFound := False;
    hlpInstallFile := InstallFiles[I];
    if IsExcludedFromConsistencyCheck(hlpInstallFile.Name, hlpExcludedFiles) = False then
    begin
      hlpFiles := FindFilesInfo(hlpInstallFile.Name, OriginalFiles);
      for J := 0 to GetArrayLength(hlpFiles) - 1 do
      begin
        if CompareStr(hlpInstallFile.Hash, hlpFiles[J].Hash) = 0 then
        begin
          hlpFound := True;
        end;
      end;

      if (GetArrayLength(hlpFiles) > 0) and (hlpFound = False) then
      begin
        CompareInfo.Add('ERR: SHA1 mismatch for file ' + hlpInstallFile.Path);
      end; 
    end;
  end;  
end;

procedure SaveResultToFile(FileName: String; CompareInfo: TStringList);
var
  hlpLog: TStringList;
  I: Integer;
begin
  Log('Consistency check result file path: ' + FileName);

  hlpLog := TStringList.Create;
  hlpLog.Add('Installation date: ' + GetDateTimeString('dd/mm/yyyy hh:nn:ss', '-', ':'));
  hlpLog.Add('Suite name: ComAp PC Suite 4.1.5.2');
  hlpLog.Add('--------------------------------------------------');
  hlpLog.AddStrings(CompareInfo);

  for I := 0 to hlpLog.Count-1 do
  begin
    SaveStringToFile(FileName, hlpLog[I] + #13#10, True);
  end;

  hlpLog.Free;
end;

procedure PerformConsistencyCheck();
var
  hlpInfoFiles, hlpCompareInfo: TStringList;
  hlpOriginalFilesInfo: TArrayOfFileInfo;
  hlpInstallFilesInfo: TArrayOfFileInfo;
  hlpLogFilePath: String;
  I: Integer;
begin
  hlpInfoFiles := TStringList.Create;
  hlpCompareInfo := TStringList.Create;
  SetArrayLength(hlpOriginalFilesInfo, 0);
  SetArrayLength(hlpInstallFilesInfo, 0);
    
  FindInfoFiles(hlpInfoFiles);
  for I := 0 to hlpInfoFiles.Count-1 do
  begin
    ParseOriginalFilesInfo(ExpandConstant('{tmp}\' + hlpInfoFiles[I]), hlpOriginalFilesInfo);
  end;

  ParseInstallFilesInfo(InstalledFilesLog(), hlpInstallFilesInfo);
  CompareFilesInfo(hlpOriginalFilesInfo, hlpInstallFilesInfo, hlpCompareInfo);
  if (hlpCompareInfo.Count > 0) then
  begin
    hlpLogFilePath := GetSetupLogDir() + 'Consistency-Check-Result.log';
    SaveResultToFile(hlpLogFilePath, hlpCompareInfo);
    SuppressibleMsgBox(FmtMessage(CustomMessage('FilesAreCorrupted'), [hlpLogFilePath]), mbCriticalError, MB_OK, IDOK);
  end;

  hlpInfoFiles.Free;
  hlpCompareInfo.Free;
end;
// end of consistencycheck

function GetLogNameFromUninstallKey(AKey: String): String;
var
  tmpPossition: Integer;
begin
  tmpPossition := Pos('\', AKey);
  if (tmpPossition <> 0) then
    Delete(AKey, 1, tmpPossition);
  StringChange(AKey, '\', '-');
  Result := AKey;
end;

function ShouldUninstallAllComApApps() : Boolean;
 // all application uninstall anouncement
begin
  if (SuppressibleMsgBox(FmtMessage(CustomMessage('UninstallAllApplications'), ['ComAp']), mbError, MB_YESNO, IDYES) = IDNO) then
  begin
    Result := False;
  end
  else
  begin
    Result := True;
  end;
end;

procedure UninstallComApApp(AName: String; AParameters: String);
 // uninstall ComAp application
var
  iResultCode: Integer;
  tmpParameters: string;
begin
  // PersistentStorage uninstall should be SILENT to show dialog about deleting data
  if Pos('PersistentStorage', AName) <> 0 then
    tmpParameters := '/SILENT' + AParameters
  else
    tmpParameters := '/VERYSILENT' + AParameters;  
  
  // protection to run exe file
  if ExtractFileExt(AName) = '.exe' then
    // uninstal one at a time
    Exec(AName, tmpParameters, '', SW_SHOW, ewWaitUntilTerminated, iResultCode);
end;

procedure UninstallAllComApApps(APlatform: String);
 // uninstall all ComAp applications
var
  tmpNames: TArrayOfString;
  iValue: Integer;
  sValue: String;
  tmpDisplayName: String;
  tmpParameters: String;
begin
  tmpParameters := tmpParameters + ' /LOG="'+ GetSetupLogDir + '%s.log"' + 
    ' /LOGDIR "' + GetSetupLogDir + '"';
  { check 64b registers for 64b win }
  if IsWin64 and (APlatform = 'x64')then
    { if key with uninstallator list exists }
    if RegGetValueNames(HKLM64, GetPcSuiteRegKey('Uninstall'), tmpNames) then
      for iValue := 0 to GetArrayLength(tmpNames) - 1 do
        { read uninstallator name and path }
        if RegQueryStringValue(HKLM64, GetPcSuiteRegKey('Uninstall'), tmpNames[iValue], sValue) then
          { odinstalovat aplikaci }
        begin
            tmpDisplayName := GetLogNameFromUninstallKey(tmpNames[iValue]);
            UninstallComApApp(sValue, Format(tmpParameters, [tmpDisplayName]));
        end;

  { check 32b registers }
  if APlatform = 'x86' then
  begin
    { if key with uninstallator list exists }
    if RegGetValueNames(HKLM32, GetPcSuiteRegKey('Uninstall'), tmpNames) then
      for iValue := 0 to GetArrayLength(tmpNames) - 1 do
        { read uninstallator name and path }
        if RegQueryStringValue(HKLM32, GetPcSuiteRegKey('Uninstall'), tmpNames[iValue], sValue) then
          { odinstalovat aplikaci }
          begin
            tmpDisplayName := GetLogNameFromUninstallKey(tmpNames[iValue]);
            UninstallComApApp(sValue, Format(tmpParameters, [tmpDisplayName]));
          end;
  end;
end;

/// <summary> Check that specific switch was passed to main package.</summary>
/// <param name="ANameOfCustomSwitch">Name of switch in format /NameOfSwitch.</param>
/// <returns>If true returns switch as string, otherwise returns empty string.</returns>
function AddSwitchIfFound(ANameOfCustomSwitch: String): String;
begin
  if FindCmdLineSwitchIndex(ANameOfCustomSwitch) <> -1 then
    Result := ' ' + ANameOfCustomSwitch
  else
    Result := '';
end;

/// <summary>Get all passed custom switches from main package and add them to input command string.</summary>
/// <param name="ACommandToPass">Command string which is updated by each present custom switch.</param>
/// <returns>Command with all custom switches.</returns>
function AddCustomSwitches(ACommandToPass: String): String;
begin
  ACommandToPass := ACommandToPass + AddSwitchIfFound('/INTERNALUSE');
  Result := ACommandToPass;
end;

function GetSubpackageParams(ASubDirectory: String): String;
 // same parameter for DIR and GROUP
 // returns parameter string used for executing inner installator
var
  sTasks: string;
  iAppLength: Integer;
begin
  Result := '/VERYSILENT';

  if FindCmdLineSwitchIndex('/SUPPRESSMSGBOXES') <> -1 then
  begin
    Result := Result + ' /SUPPRESSMSGBOXES';
  end;

  if ASubDirectory <> '' then
  begin
    Result := Result + ' /DIR="' + WizardDirValue + '\' + ASubDirectory + '"';
    Result := Result + ' /GROUP="' + WizardGroupValue + '\' + ASubDirectory + '"'; 
  end;
  
  { task list }
  sTasks := '';
  if IsTaskSelected('desktopicon') then
  begin
    sTasks := sTasks + ',desktopicon';
  end;
  if IsTaskSelected('commoninstall') then
  begin
    sTasks := sTasks + ',commoninstall';
  end;
  if IsTaskSelected('userinstall') then
  begin
    sTasks := sTasks + ',userinstall';
  end;

  if Length(sTasks) > 0 then
  begin
    sTasks := Copy(sTasks, 2, Length(sTasks) - 1);
  end;

  Result := Result + ' /TASKS="' + sTasks + '"';
   // always returning
  iAppLength := Length('{app}');  
  Result := Result + ' /regpub "ComAp" /regpcsuite "ComAp PC Suite"'
   // pass register key names

  { logging }
  Result := Result + ' /LOG="' + csSetupLogFileName + '"';
  Result := Result + ' /LOGDIR "' + GetSetupLogDir + '"';

  Result := AddCustomSwitches(Result);
  
end;

{ event for showing required value of progress bar in wizard form COMAPPCSUITE-169 }
procedure UpdateProgress(Position: Integer);
begin
  WizardForm.ProgressGauge.Position := Position * WizardForm.ProgressGauge.Max div 100;
  if (Position = 100) then
    Sleep(1000);
   // show last step for a while
end;

procedure ProcessLogAndUpdateProgress(ProgressPosition: Integer; SetupFileName: String);
var
  tmpLogFilePath : String;
begin
  tmpLogFilePath := ExpandConstant('{tmp}\' + csSetupLogFileName);
  BackupLogFile(tmpLogFilePath, ChangeFileExt(SetupFileName, '.log'));
  ProcessSetupLog(tmpLogFilePath);
  UpdateProgress(ProgressPosition);
end;

procedure CurPageChanged(CurPageID: Integer);
begin
  { finally order directory by name }
  if (CurPageID = wpFinished) then
  begin
    SortStartMenuFolderByName('ComAp PC Suite');
    BackUpUserDirs();
  end;

end;

{ Event handler on next button click. Base function. }
function NextButtonClick(CurPageID: Integer): Boolean;
begin
  
  Result := True;
end;

function ShouldSkipPage(PageID: Integer): Boolean;
begin
  Result := False;

end;

function InitializeSetup(): Boolean;
begin
  Log(Format('Temp log file is created at path: %s', [ExpandConstant('{log}')]));
  BackupMainLogFile();
  Result := True;
  PsiPackageCanRun := True;
  Result := Result and CheckWindowsVersion($05010000,FmtMessage(CustomMessage('NotSupportOS'),['InteliMonitor', 'Windows XP']));
  Result := Result and CheckActualVersion();
end;

procedure DeinitializeSetup();
begin
  BackupMainLogFile(); 
end;

procedure InitializeWizard();
begin
  
end;

function InitializeUninstall(): Boolean;
begin
  Result := True
  Result := ShouldUninstallAllComApApps(); 
end;

// Gets list of file from set directory
procedure GetFilesInDir(const Directory: String; Files: TStringList);
var
  tmpFindRec: TFindRec;
begin
  if FindFirst(ExpandConstant(Directory), tmpFindRec) then
  try
    repeat
      if tmpFindRec.Attributes and FILE_ATTRIBUTE_DIRECTORY = 0 then
        Files.Add(tmpFindRec.Name);
    until
      not FindNext(tmpFindRec);
  finally
    FindClose(tmpFindRec);
  end;
end;

// Checks if folder is empty 
function IsFolderEmpty(const Directory: String): Boolean;
var
  tmpFiles: TStringList;
begin
  tmpFiles := TStringList.Create;
  try
	GetFilesInDir(Directory + '\*', tmpFiles);
	Result := tmpFiles.Count = 0;
  finally
	tmpFiles.Free;
  end;
end;

// Uninstall old product for installation suite package
procedure UninstallOldProduct(UninstallDirPath, UninstallParameters: String);
var
  iResultCode, iIdx: Integer;
  tmpFiles: TStringList;
  sUninstallExeFilePath, sUninstallDatFilePath: String;
begin
  tmpFiles := TStringList.Create;
  try
  begin
    GetFilesInDir(UninstallDirPath + '\unins????.exe', tmpFiles);
	if (tmpFiles.Count = 0) then
	  Exit;

    for iIdx := tmpFiles.Count - 1 downto 0 do
	begin
	  sUninstallExeFilePath := ExpandConstant(UninstallDirPath) + '\' + tmpFiles[iIdx];
	  sUninstallDatFilePath := Copy(sUninstallExeFilePath, 1, Length(sUninstallExeFilePath) - 3) + 'dat';
	end;

	if (FileExists(sUninstallExeFilePath) and FileExists(sUninstallDatFilePath)) then
	begin
	  Exec(sUninstallExeFilePath, UninstallParameters,'', SW_HIDE, ewWaitUntilTerminated, iResultCode);
	end
	else if (FileExists(sUninstallExeFilePath) and (not FileExists(sUninstallDatFilePath))) then
	begin
	  try
		DeleteFile(sUninstallExeFilePath);
		if (IsFolderEmpty(UninstallDirPath)) then
		  DelTree(ExpandConstant(UninstallDirPath), True, True, False);
	  except
	  end;
	end;
  end;
  finally
	tmpFiles.Free;
  end;
end;

{ Delete application folder after unistallation if is empty. }
function ShouldDeletePcSuiteDirectory() : Boolean;
begin
      Result := False;
      if DirExists(ExpandConstant('{app}')) and (IsFolderEmpty(ExpandConstant('{app}'))) then
      begin
          Result := True;
      end;
end;

{ Built in event for tasks which happened after uninstallation. }
procedure DeinitializeUninstall();
begin
  if ShouldDeletePcSuiteDirectory() then
    begin
       DelTree(ExpandConstant('{app}'), True, True, True);
    end;
end;

procedure CurStepChanged(CurStep: TSetupStep);
begin
  case CurStep of
    ssInstall:
      begin
        
      end;
    ssPostInstall:
      begin
        PerformConsistencyCheck();
      end;
  end;
end;

procedure CurUninstallStepChanged(CurUninstallStep: TUninstallStep);
begin
  case CurUninstallStep of
    usUninstall:
      begin
        UninstallAllComApApps('x86');
      end;
    usPostUninstall:
      begin
        
      end;
  end;
end;

function NeedRestart(): Boolean;
begin
  Result := RestartIsNeeded;
end; 


//; SharedSetup
[Setup]
AppId={{347EEEAE-0F1E-4823-BE31-E3F0C6B8960C}
AppName="ComAp PC Suite"
AppVerName="ComAp PC Suite 4.1.5.2"
AppPublisher="ComAp a.s."
AppVersion="4.1.5.2"
AppCopyright="Copyright (C) 2022 ComAp a.s."
VersionInfoVersion="4.1.5.2"
AppPublisherURL="http://www.comap.cz/"
AppSupportURL="http://www.comap.cz/"
AppUpdatesURL="http://www.comap.cz/"
DefaultDirName="{pf}\ComAp PC Suite"
DefaultGroupName="ComAp PC Suite"
DisableWelcomePage=no
DisableDirPage=auto
DisableProgramGroupPage=auto
Compression=lzma
SolidCompression=yes
WizardResizable=yes
WizardSizePercent=100
WizardStyle=modern
WizardImageFile=compiler:WizClassicImage-IS.bmp
WizardSmallImageFile=compiler:WizClassicSmallImage-IS.bmp
UninstallDisplayIcon="{app}\ComAp.ico"
OutputDir="C:\bamboo-agent\bamboo-agent-home\xml-data\build-dir\IM-IMIP62-JOB1\Release\InstallationPackages"
OutputBaseFilename="InteliMonitor-Product-Suite-4.1.5.2"
SetupLogging=yes
ChangesAssociations=yes
SignedUninstaller=yes
RestartIfNeededByRun=yes
ShowLanguageDialog=no


[Types]
Name: "full"; Description: {cm:InstallationTypeFull}
Name: "compact"; Description: {cm:InstallationTypeCompact}
Name: "custom"; Description: {cm:InstallationTypeCustom}; Flags: iscustom

[Components]
Name: "InteliMonitor"; Description: "InteliMonitor 4.1.5.3"; Types: compact custom full; Flags: disablenouninstallwarning fixed
Name: "InteliMonitor/PLCMonitor"; Description: "PLCMonitor 2.1.3.2"; Types: compact custom full; Flags: disablenouninstallwarning


[Tasks]
Name: "commoninstall"; Description: {cm:ForAllUsers}; GroupDescription: {cm:InstallDataFor}; Flags: exclusive
Name: "userinstall"; Description: {cm:ForTheCurrentUserOnly}; GroupDescription: {cm:InstallDataFor}; Flags: exclusive unchecked
Name: "desktopicon"; Description: {cm:CreateDesktopIcon}; GroupDescription: {cm:AdditionalIcons}


[Dirs]
Name: "{commonappdata}\ComAp PC Suite\InteliMonitor"; Permissions: users-modify


[Files]
Source: "C:\bamboo-agent\bamboo-agent-home\xml-data\build-dir\IM-IMIP62-JOB1\Release\Temp\AutomaticBuildResults\PLCMonitor\2.1.3.2\InstallationPackages\PLCMonitor-Install.exe"; DestDir: "{tmp}"; Components: InteliMonitor/PLCMonitor
Source: "C:\bamboo-agent\bamboo-agent-home\xml-data\build-dir\IM-IMIP62-JOB1\Release\Temp\AutomaticBuildResults\PLCEditor\1.0.0.1\DataPackages\PLCEditor-Data.igc"; DestDir: "{tmp}"; Components: InteliMonitor/PLCMonitor
Source: "C:\bamboo-agent\bamboo-agent-home\xml-data\build-dir\IM-IMIP62-JOB1\Release\Temp\AutomaticBuildResults\InteliMonitor\4.1.5.3\InstallationPackages\InteliMonitor-Install.exe"; DestDir: "{tmp}"; Components: InteliMonitor
Source: "C:\bamboo-agent\bamboo-agent-home\xml-data\build-dir\IM-IMIP62-JOB1\Release\Temp\ManualBuildResults\DefaultSites\1.0.0.0\DataPackages\DefaultSites.igc"; DestDir: "{tmp}"; Components: InteliMonitor
Source: "C:\bamboo-agent\bamboo-agent-home\xml-data\build-dir\IM-IMIP62-JOB1\Release\Temp\ManualBuildResults\IGSNTCurves\1.0.0.0\IGSNTCurves.igc"; DestDir: "{tmp}"; Components: InteliMonitor
Source: "C:\bamboo-agent\bamboo-agent-home\xml-data\build-dir\IM-IMIP62-JOB1\Release\Temp\AutomaticBuildResults\IndexFiles\1.1.0.64\DataPackages\IndexFiles.igc"; DestDir: "{tmp}"; Components: InteliMonitor
Source: "C:\bamboo-agent\bamboo-agent-home\xml-data\build-dir\IM-IMIP62-JOB1\Release\Temp\AutomaticBuildResults\XmlDatabase\1.6.0.10\DataPackages\DriveConfig.idc"; DestDir: "{tmp}"; Components: InteliMonitor
Source: "C:\bamboo-agent\bamboo-agent-home\xml-data\build-dir\IM-IMIP62-JOB1\Release\Temp\AutomaticBuildResults\XmlDatabase\1.6.0.10\DataPackages\GenConfig.igc"; DestDir: "{tmp}"; Components: InteliMonitor
Source: "C:\bamboo-agent\bamboo-agent-home\xml-data\build-dir\IM-IMIP62-JOB1\Release\Temp\AutomaticBuildResults\XmlDatabase\1.6.0.10\DataPackages\InteliMonitor.igc"; DestDir: "{tmp}"; Components: InteliMonitor
Source: "C:\bamboo-agent\bamboo-agent-home\xml-data\build-dir\IM-IMIP62-JOB1\Release\Temp\AutomaticBuildResults\XmlDatabase\1.6.0.10\DataPackages\LiteEdit.iwe"; DestDir: "{tmp}"; Components: InteliMonitor
Source: "C:\bamboo-agent\bamboo-agent-home\xml-data\build-dir\IM-IMIP62-JOB1\Release\Temp\AutomaticBuildResults\XmlDatabase\1.6.0.10\DataPackages\SharedXmlDelphi.igc"; DestDir: "{tmp}"; Components: InteliMonitor
Source: "C:\bamboo-agent\bamboo-agent-home\xml-data\build-dir\IM-IMIP62-JOB1\Release\Temp\AutomaticBuildResults\XmlDatabase\1.6.0.10\DataPackages\InteliVision.ixc"; DestDir: "{tmp}"; Components: InteliMonitor
Source: "C:\bamboo-agent\bamboo-agent-home\xml-data\build-dir\IM-IMIP62-JOB1\Release\Temp\AutomaticBuildResults\InteliDDE\4.12.4.2\InstallationPackages\InteliDDE-Install.exe"; DestDir: "{tmp}"; Components: InteliMonitor
Source: "C:\bamboo-agent\bamboo-agent-home\xml-data\build-dir\IM-IMIP62-JOB1\Release\Temp\AutomaticBuildResults\FirmwareImport\2.0.0.1\InstallationPackages\FirmwareImport-Install.exe"; DestDir: "{tmp}"; Components: InteliMonitor
Source: "C:\bamboo-agent\bamboo-agent-home\xml-data\build-dir\IM-IMIP62-JOB1\Release\Temp\Licence.rtf"; DestDir: "{app}"; Languages: English
Source: "C:\bamboo-agent\bamboo-agent-home\xml-data\build-dir\IM-IMIP62-JOB1\Release\Temp\LicenceJapanese.rtf"; DestDir: "{app}"; Languages: Japanese
Source: "C:\bamboo-agent\bamboo-agent-home\xml-data\build-dir\IM-IMIP62-JOB1\Release\Temp\ComAp.ico"; DestDir: "{app}"


[Icons]
Name: "{group}\{cm:UninstallProgram,ComAp PC Suite}"; Filename: "{uninstallexe}"


[Languages]
Name: "english"; MessagesFile: "C:\bamboo-agent\bamboo-agent-home\xml-data\build-dir\IM-IMIP62-JOB1\Temp\English.isl"; LicenseFile: "C:\bamboo-agent\bamboo-agent-home\xml-data\build-dir\IM-IMIP62-JOB1\Release\Temp\Licence.rtf"
Name: "Japanese"; MessagesFile: "C:\bamboo-agent\bamboo-agent-home\xml-data\build-dir\IM-IMIP62-JOB1\Temp\Japanese.isl"; LicenseFile: "C:\bamboo-agent\bamboo-agent-home\xml-data\build-dir\IM-IMIP62-JOB1\Release\Temp\LicenceJapanese.rtf"

[Registry]
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|}"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|}. =  with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|}"; ValueType: string; ValueName: "Install Dir"; ValueData: "{app}"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|}.Install Dir = {app} with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|InteliMonitor}"; ValueType: string; ValueName: "MainInstallationPackage"; ValueData: "InteliMonitor-Product-Suite"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|InteliMonitor}.MainInstallationPackage = InteliMonitor-Product-Suite with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|InteliMonitor}"; ValueType: string; ValueName: "MainInstallationPackageVersion"; ValueData: "4.1.5.2"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|InteliMonitor}.MainInstallationPackageVersion = 4.1.5.2 with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|InteliMonitor}"; ValueType: string; ValueName: "MainInstallationPackageSetupLog"; ValueData: "{code:GetSetupLogDirProxy}\InteliMonitor-Product-Suite.log"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|InteliMonitor}.MainInstallationPackageSetupLog = {code:GetSetupLogDirProxy}\InteliMonitor-Product-Suite.log with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|DefaultSites}"; ValueType: string; ValueName: "MainInstallationPackage"; ValueData: "InteliMonitor-Product-Suite"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|DefaultSites}.MainInstallationPackage = InteliMonitor-Product-Suite with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|DefaultSites}"; ValueType: string; ValueName: "MainInstallationPackageVersion"; ValueData: "4.1.5.2"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|DefaultSites}.MainInstallationPackageVersion = 4.1.5.2 with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|DefaultSites}"; ValueType: string; ValueName: "MainInstallationPackageSetupLog"; ValueData: "{code:GetSetupLogDirProxy}\InteliMonitor-Product-Suite.log"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|DefaultSites}.MainInstallationPackageSetupLog = {code:GetSetupLogDirProxy}\InteliMonitor-Product-Suite.log with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|IGSNTCurves}"; ValueType: string; ValueName: "MainInstallationPackage"; ValueData: "InteliMonitor-Product-Suite"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|IGSNTCurves}.MainInstallationPackage = InteliMonitor-Product-Suite with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|IGSNTCurves}"; ValueType: string; ValueName: "MainInstallationPackageVersion"; ValueData: "4.1.5.2"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|IGSNTCurves}.MainInstallationPackageVersion = 4.1.5.2 with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|IGSNTCurves}"; ValueType: string; ValueName: "MainInstallationPackageSetupLog"; ValueData: "{code:GetSetupLogDirProxy}\InteliMonitor-Product-Suite.log"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|IGSNTCurves}.MainInstallationPackageSetupLog = {code:GetSetupLogDirProxy}\InteliMonitor-Product-Suite.log with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|IndexFiles}"; ValueType: string; ValueName: "MainInstallationPackage"; ValueData: "InteliMonitor-Product-Suite"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|IndexFiles}.MainInstallationPackage = InteliMonitor-Product-Suite with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|IndexFiles}"; ValueType: string; ValueName: "MainInstallationPackageVersion"; ValueData: "4.1.5.2"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|IndexFiles}.MainInstallationPackageVersion = 4.1.5.2 with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|IndexFiles}"; ValueType: string; ValueName: "MainInstallationPackageSetupLog"; ValueData: "{code:GetSetupLogDirProxy}\InteliMonitor-Product-Suite.log"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|IndexFiles}.MainInstallationPackageSetupLog = {code:GetSetupLogDirProxy}\InteliMonitor-Product-Suite.log with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|XmlDatabase}"; ValueType: string; ValueName: "MainInstallationPackage"; ValueData: "InteliMonitor-Product-Suite"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|XmlDatabase}.MainInstallationPackage = InteliMonitor-Product-Suite with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|XmlDatabase}"; ValueType: string; ValueName: "MainInstallationPackageVersion"; ValueData: "4.1.5.2"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|XmlDatabase}.MainInstallationPackageVersion = 4.1.5.2 with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|XmlDatabase}"; ValueType: string; ValueName: "MainInstallationPackageSetupLog"; ValueData: "{code:GetSetupLogDirProxy}\InteliMonitor-Product-Suite.log"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|XmlDatabase}.MainInstallationPackageSetupLog = {code:GetSetupLogDirProxy}\InteliMonitor-Product-Suite.log with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|InteliDDE}"; ValueType: string; ValueName: "MainInstallationPackage"; ValueData: "InteliMonitor-Product-Suite"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|InteliDDE}.MainInstallationPackage = InteliMonitor-Product-Suite with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|InteliDDE}"; ValueType: string; ValueName: "MainInstallationPackageVersion"; ValueData: "4.1.5.2"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|InteliDDE}.MainInstallationPackageVersion = 4.1.5.2 with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|InteliDDE}"; ValueType: string; ValueName: "MainInstallationPackageSetupLog"; ValueData: "{code:GetSetupLogDirProxy}\InteliMonitor-Product-Suite.log"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|InteliDDE}.MainInstallationPackageSetupLog = {code:GetSetupLogDirProxy}\InteliMonitor-Product-Suite.log with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|FirmwareImport}"; ValueType: string; ValueName: "MainInstallationPackage"; ValueData: "InteliMonitor-Product-Suite"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|FirmwareImport}.MainInstallationPackage = InteliMonitor-Product-Suite with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|FirmwareImport}"; ValueType: string; ValueName: "MainInstallationPackageVersion"; ValueData: "4.1.5.2"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|FirmwareImport}.MainInstallationPackageVersion = 4.1.5.2 with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|FirmwareImport}"; ValueType: string; ValueName: "MainInstallationPackageSetupLog"; ValueData: "{code:GetSetupLogDirProxy}\InteliMonitor-Product-Suite.log"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|FirmwareImport}.MainInstallationPackageSetupLog = {code:GetSetupLogDirProxy}\InteliMonitor-Product-Suite.log with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|PLCMonitor}"; ValueType: string; ValueName: "MainInstallationPackage"; ValueData: "InteliMonitor-Product-Suite"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|PLCMonitor}.MainInstallationPackage = InteliMonitor-Product-Suite with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|PLCMonitor}"; ValueType: string; ValueName: "MainInstallationPackageVersion"; ValueData: "4.1.5.2"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|PLCMonitor}.MainInstallationPackageVersion = 4.1.5.2 with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|PLCMonitor}"; ValueType: string; ValueName: "MainInstallationPackageSetupLog"; ValueData: "{code:GetSetupLogDirProxy}\InteliMonitor-Product-Suite.log"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|PLCMonitor}.MainInstallationPackageSetupLog = {code:GetSetupLogDirProxy}\InteliMonitor-Product-Suite.log with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|PLCEditor}"; ValueType: string; ValueName: "MainInstallationPackage"; ValueData: "InteliMonitor-Product-Suite"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|PLCEditor}.MainInstallationPackage = InteliMonitor-Product-Suite with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|PLCEditor}"; ValueType: string; ValueName: "MainInstallationPackageVersion"; ValueData: "4.1.5.2"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|PLCEditor}.MainInstallationPackageVersion = 4.1.5.2 with flags = uninsdeletekey')
Root: HKLM; Subkey: "{code:GetPcSuiteRegKey|PLCEditor}"; ValueType: string; ValueName: "MainInstallationPackageSetupLog"; ValueData: "{code:GetSetupLogDirProxy}\InteliMonitor-Product-Suite.log"; Flags: uninsdeletekey; BeforeInstall: LogRegistry('HKLM\{code:GetPcSuiteRegKey|PLCEditor}.MainInstallationPackageSetupLog = {code:GetSetupLogDirProxy}\InteliMonitor-Product-Suite.log with flags = uninsdeletekey')


[Run]
Filename: "{tmp}\PLCMonitor-Install.exe"; Parameters: "{code:GetSubpackageParams|Tools\PLCMonitor}"; Flags: skipifdoesntexist runhidden; StatusMsg: {cm:Installing,PLCMonitor-Install.exe}; BeforeInstall: UpdateProgress(0); AfterInstall: ProcessLogAndUpdateProgress(7, 'PLCMonitor-Install.exe')
Filename: "{tmp}\InteliMonitor-Install.exe"; Parameters: "{code:GetSubpackageParams|InteliMonitor}"; Flags: skipifdoesntexist runhidden; StatusMsg: {cm:Installing,InteliMonitor-Install.exe}; AfterInstall: ProcessLogAndUpdateProgress(14, 'InteliMonitor-Install.exe')
Filename: "{tmp}\InteliDDE-Install.exe"; Parameters: "{code:GetSubpackageParams|Tools\InteliDDE}"; Flags: skipifdoesntexist runhidden; StatusMsg: {cm:Installing,InteliDDE-Install.exe}; AfterInstall: ProcessLogAndUpdateProgress(21, 'InteliDDE-Install.exe')
Filename: "{tmp}\FirmwareImport-Install.exe"; Parameters: "{code:GetSubpackageParams|Tools\FirmwareImport}"; Flags: skipifdoesntexist runhidden; StatusMsg: {cm:Installing,FirmwareImport-Install.exe}; AfterInstall: ProcessLogAndUpdateProgress(28, 'FirmwareImport-Install.exe')
Filename: "{tmp}\PLCEditor-Data.igc"; Verb: install; Flags: shellexec waituntilterminated skipifdoesntexist runhidden; StatusMsg: {cm:Installing,PLCEditor-Data.igc}; AfterInstall: UpdateProgress(35)
Filename: "{tmp}\DefaultSites.igc"; Verb: install; Flags: shellexec waituntilterminated skipifdoesntexist runhidden; StatusMsg: {cm:Installing,DefaultSites.igc}; AfterInstall: UpdateProgress(42)
Filename: "{tmp}\IGSNTCurves.igc"; Verb: install; Flags: shellexec waituntilterminated skipifdoesntexist runhidden; StatusMsg: {cm:Installing,IGSNTCurves.igc}; AfterInstall: UpdateProgress(50)
Filename: "{tmp}\IndexFiles.igc"; Verb: install; Flags: shellexec waituntilterminated skipifdoesntexist runhidden; StatusMsg: {cm:Installing,IndexFiles.igc}; AfterInstall: UpdateProgress(57)
Filename: "{tmp}\DriveConfig.idc"; Verb: install; Flags: shellexec waituntilterminated skipifdoesntexist runhidden; StatusMsg: {cm:Installing,DriveConfig.idc}; AfterInstall: UpdateProgress(64)
Filename: "{tmp}\GenConfig.igc"; Verb: install; Flags: shellexec waituntilterminated skipifdoesntexist runhidden; StatusMsg: {cm:Installing,GenConfig.igc}; AfterInstall: UpdateProgress(71)
Filename: "{tmp}\InteliMonitor.igc"; Verb: install; Flags: shellexec waituntilterminated skipifdoesntexist runhidden; StatusMsg: {cm:Installing,InteliMonitor.igc}; AfterInstall: UpdateProgress(78)
Filename: "{tmp}\LiteEdit.iwe"; Verb: install; Flags: shellexec waituntilterminated skipifdoesntexist runhidden; StatusMsg: {cm:Installing,LiteEdit.iwe}; AfterInstall: UpdateProgress(85)
Filename: "{tmp}\SharedXmlDelphi.igc"; Verb: install; Flags: shellexec waituntilterminated skipifdoesntexist runhidden; StatusMsg: {cm:Installing,SharedXmlDelphi.igc}; AfterInstall: UpdateProgress(92)
Filename: "{tmp}\InteliVision.ixc"; Verb: install; Flags: shellexec waituntilterminated skipifdoesntexist runhidden; StatusMsg: {cm:Installing,InteliVision.ixc}; AfterInstall: UpdateProgress(100)


[InstallDelete]


[UninstallDelete]


